<?php
/**
 * Premium Core plugin class.
 *
 * @link  http://www.boldgrid.com
 * @since 1.0.0
 *
 * @package    Boldgrid_Backup_Premium
 * @subpackage Boldgrid_Backup_Premium/includes
 */

/**
 * The premium core plugin class.
 *
 * This is used to define internationalization and admin-specific hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Boldgrid_Backup_Premium
 * @subpackage Boldgrid_Backup_Premium/includes
 * @author     BoldGrid.com <wpb@boldgrid.com>
 */
class Boldgrid_Backup_Premium {

	/**
	 * The core class object.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    Boldgrid_Backup_Admin_Core
	 */
	private $core;

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    Boldgrid_Backup_Premium_Loader $loader
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    string $plugin_name
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    string $version The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area of the site.
	 *
	 * @since 1.0.0
	 *
	 * @param Boldgrid_Backup_Admin_Core $core
	 */
	public function __construct( $core ) {
		$this->plugin_name = 'boldgrid-backup-premium';
		$this->version = ( defined( 'BOLDGRID_BACKUP_PREMIUM_VERSION' ) ? BOLDGRID_BACKUP_PREMIUM_VERSION : '' );
		$this->core = $core;

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Boldgrid_Backup_Premium_Loader. Orchestrates the hooks of the plugin.
	 * - Boldgrid_Backup_Premium_i18n. Defines internationalization functionality.
	 * - Boldgrid_Backup_Premium_Admin. Defines all hooks for the admin area.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since  1.0.0
	 * @access private
	 */
	private function load_dependencies() {
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/vendor/autoload.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/includes/class-boldgrid-backup-premium-loader.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/includes/class-boldgrid-backup-premium-i18n.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/class-boldgrid-backup-premium-core.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/class-boldgrid-backup-premium-admin-archive-browser.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/class-boldgrid-backup-premium-admin-historical.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/class-boldgrid-backup-premium-admin-plugin-editor.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/class-boldgrid-backup-premium-admin-history.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/class-boldgrid-backup-premium-admin-recent.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/remote/amazon_s3.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/remote/amazon_s3_backups_page.php';
		require_once BOLDGRID_BACKUP_PREMIUM_PATH . '/admin/remote/amazon_s3_bucket.php';

		$this->loader = new Boldgrid_Backup_Premium_Loader();
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Boldgrid_Backup_Premium_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since  1.0.0
	 * @access private
	 */
	private function set_locale() {
		$plugin_i18n = new Boldgrid_Backup_Premium_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
	}

	/**
	 * Register all of the hooks related to the admin area functionality of the plugin.
	 *
	 * @since  1.0.0
	 * @access private
	 */
	private function define_admin_hooks() {
		$premium_core = new Boldgrid_Backup_Premium_Admin_Core( $this->core );

		/*
		 * Load amazon S3.
		 *
		 * Commented filters are required for any 3rd party storage provider to
		 * register themselves with BoldGrid Backup. The additional filters are
		 * miscellaneous and are specific to our Amazon S3 implementation.
		 */
		// When viewing details of an individual backup:
		// # Allow one click upload.
		$this->loader->add_action( 'boldgrid_backup_single_archive_remote_options', $premium_core->amazon_s3, 'single_archive_remote_option' );
		// # Process the one click upload.
		$this->loader->add_action( 'wp_ajax_boldgrid_backup_remote_storage_upload_amazon_s3', $premium_core->amazon_s3, 'ajax_upload' );
		// After a backup file has been created, add remote provider to jobs queue.
		$this->loader->add_action( 'boldgrid_backup_post_archive_files',            $premium_core->amazon_s3, 'post_archive_files' );
		// This is the filter executed by the jobs queue.
		$this->loader->add_filter( 'boldgrid_backup_amazon_s3_upload_post_archive', $premium_core->amazon_s3, 'upload_post_archiving' );
		// On the settings page:
		// # List this storage provider.
		$this->loader->add_filter( 'boldgrid_backup_register_storage_location',     $premium_core->amazon_s3, 'register_storage_location' );
		// # Refresh the configure / configured notice.
		$this->loader->add_action( 'wp_ajax_boldgrid_backup_is_setup_amazon_s3',    $premium_core->amazon_s3, 'is_setup_ajax' );
		$this->loader->add_action( 'admin_menu',                                    $premium_core->amazon_s3, 'add_menu_items' );
		$this->loader->add_filter( 'boldgrid_backup_backup_locations',           $premium_core->amazon_s3->backups_page, 'backup_locations', 10, 2 );
		$this->loader->add_filter( 'admin_enqueue_scripts',                      $premium_core->amazon_s3->backups_page, 'admin_enqueue_scripts' );
		$this->loader->add_action( 'wp_ajax_boldgrid_backup_remote_storage_download_amazon_s3', $premium_core->amazon_s3->backups_page, 'wp_ajax_download' );
		$this->loader->add_action( 'boldgrid_backup_get_all', $premium_core->amazon_s3, 'filter_get_all' );

		$this->loader->add_filter( 'boldgrid_backup_file_actions', $premium_core->archive_browser, 'wp_ajax_file_actions', 20, 2 );
		$this->loader->add_action( 'boldgrid_backup_enqueue_archive_details', $premium_core->archive_browser, 'enqueue_archive_details' );
		$this->loader->add_action( 'wp_ajax_boldgrid_backup_restore_single_file', $premium_core->archive_browser, 'wp_ajax_restore_file' );

		$this->loader->add_filter( 'boldgrid_backup_create_dir_config', $premium_core->historical, 'create_dir_config', 10, 2 );
		$this->loader->add_action( 'admin_menu', $premium_core->historical, 'add_menu_items' );
		$this->loader->add_action( 'admin_enqueue_scripts', $premium_core->historical, 'admin_enqueue_scripts' );
		$this->loader->add_action( 'wp_ajax_boldgrid_backup_restore_historical', $premium_core->historical, 'wp_ajax_restore_historical' );
		$this->loader->add_action( 'wp_ajax_boldgrid_backup_get_historical_versions', $premium_core->historical, 'wp_ajax_get_historical_versions' );

		$this->loader->add_action( 'admin_enqueue_scripts', $premium_core->plugin_editor, 'admin_enqueue_scripts' );
		$this->loader->add_action( 'wp_ajax_boldgrid_backup_save_copy', $premium_core->plugin_editor, 'wp_ajax_save_copy' );

		$this->loader->add_action( 'boldgrid_backup_add_history', $premium_core->history, 'add', 10, 2 );
		$this->loader->add_action( 'upgrader_process_complete', $premium_core->history, 'upgrader_process_complete', 10, 2 );
		$this->loader->add_action( 'update_option_active_plugins', $premium_core->history, 'update_option_active_plugins', 10, 3 );
		$this->loader->add_action( 'delete_plugin', $premium_core->history, 'delete_plugin' );
		$this->loader->add_action( 'switch_theme', $premium_core->history, 'switch_theme', 10, 3 );
		$this->loader->add_action( 'boldgrid_backup_post_archive_files', $premium_core->history, 'post_archive_files' );
		$this->loader->add_action( 'boldgrid_backup_retention_deleted', $premium_core->history, 'retention_deleted' );
		$this->loader->add_action( 'boldgrid_backup_user_deleted_backup', $premium_core->history, 'user_deleted_backup', 10, 2 );
		$this->loader->add_action( 'boldgrid_backup_settings_updated', $premium_core->history, 'settings_updated' );
		$this->loader->add_action( 'boldgrid_backup_remote_uploaded', $premium_core->history, 'remote_uploaded', 10, 2 );
		$this->loader->add_action( 'boldgrid_backup_remote_retention_deleted', $premium_core->history, 'remote_retention_deleted', 10, 2 );
		$this->loader->add_filter( 'boldgrid_backup_tools_sections', $premium_core->history, 'filter_tools_section' );

		$this->loader->add_filter( 'boldgrid_backup_tools_sections', $premium_core->recent, 'filter_tools_section' );
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since 1.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since  1.0.0
	 * @return string The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since  1.0.0
	 * @return Boldgrid_Backup_Premium_Loader Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since  1.0.0
	 * @return string The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}
}
